#include "m68000.h"
#include "driver.h"
#include "cpuintrf.h"

static int pending_interrupts;

struct Cyclone cyclone;

static unsigned char MyRead8(unsigned int a) {
	return cpu_readmem24((a)&0xffffff);	
}

static unsigned short MyRead16(unsigned int a) {
	return cpu_readmem24_word((a)&0xffffff);
}

static unsigned int MyRead32(unsigned int a) {
	return cpu_readmem24_dword((a)&0xffffff);
}

static void MyWrite8(unsigned int a,unsigned char d) {
	cpu_writemem24((a)&0xffffff,d);
}

static void MyWrite16(unsigned int a,unsigned short d) {
	cpu_writemem24_word((a)&0xffffff,d);
}

static void MyWrite32(unsigned int a,unsigned int d) {
	cpu_writemem24_dword((a)&0xffffff,d);
}

static unsigned int MyCheckPc(unsigned int pc)
{
  	pc = (pc-cyclone.membase) & 0xffffff; /* Get the real program counter */
	change_pc24(pc);
	cyclone.membase=(unsigned int)OP_ROM;
  	return cyclone.membase+pc; /* New program counter */
}

static void MyIrqCallback(int irq)
{
	cyclone.irq=0;
	if(pending_interrupts & 0xff000000)
	{
		int level, mask = 0x80000000;
		for (level = 7; level; level--, mask >>= 1)
			if (pending_interrupts & mask)
				break;
   		if(level>cyclone.irq)
   		{
   			pending_interrupts &= ~(1 << (level + 24));
   			cyclone.irq = level;
   		}
	}
}

void MC68000_Reset(void)
{
	asm_memset(&cyclone, 0,sizeof(cyclone));
	cyclone.checkpc=MyCheckPc;
	cyclone.read8  =MyRead8;
	cyclone.read16 =MyRead16;
	cyclone.read32 =MyRead32;
	cyclone.write8 =MyWrite8;
	cyclone.write16=MyWrite16;
	cyclone.write32=MyWrite32;
	cyclone.fetch8  =MyRead8;
	cyclone.fetch16 =MyRead16;
	cyclone.fetch32 =MyRead32;
	cyclone.IrqCallback=MyIrqCallback; /* 0 */
  	cyclone.srh=0x27; /* Set supervisor mode */
  	cyclone.a[7]=cyclone.read32(0); /* Get Stack Pointer */
  	cyclone.membase=0;
  	cyclone.pc=cyclone.checkpc(cyclone.read32(4)); /* Get Program Counter */
   	pending_interrupts = 0;
   	CycloneInit();
}

int  MC68000_GetPC(void)
{
	return (cyclone.pc - cyclone.membase) & 0xffffff;
}

void MC68000_SetRegs(MC68000_Regs *src)
{
	asm_memcpy(&cyclone,&(src->regs),sizeof(cyclone));
	pending_interrupts = src->pending_interrupts;
}

void MC68000_GetRegs(MC68000_Regs *dst)
{
	asm_memcpy(&(dst->regs),&cyclone,sizeof(cyclone));
	dst->pending_interrupts = pending_interrupts;
}


void MC68000_Cause_Interrupt(int level)
{
	if (level >= 1 && level <= 7)
		pending_interrupts |= 1 << (24 + level);
}


void MC68000_Clear_Pending_Interrupts(void)
{
	pending_interrupts &= ~0xff000000;
}


int MC68000_Execute(int cycles)
{
	if ((pending_interrupts & MC68000_STOP) && !(pending_interrupts & 0xff000000))
		return cycles;

	if (pending_interrupts & 0xff000000) 
	{
		int level, mask = 0x80000000;
		for (level = 7; level; level--, mask >>= 1)
			if (pending_interrupts & mask)
				break;
		if(level>cyclone.irq)
		{
			pending_interrupts &= ~(1 << (level + 24));
			cyclone.irq = level;
		}
	}

	cyclone.cycles = cycles;
	{
		extern int previouspc;
		previouspc = (cyclone.pc - cyclone.membase) & 0xffffff;
	}
	CycloneRun(&cyclone);
   	return (cycles - cyclone.cycles);
}
